# coding=utf-8

import globalPluginHandler
import addonHandler
import scriptHandler
from .audioManager import AudioManager
from .audioNavigator import PlaybackDeviceNavigator, RecordingDeviceNavigator, SessionNavigator, NVDAOutputDeviceNavigator
import ui
import wx
import config
import os

addonHandler.initTranslation()

CATEGORY_NAME = _("Audio Manager")

class GlobalPlugin(globalPluginHandler.GlobalPlugin):
    def __init__(self):
        super(globalPluginHandler.GlobalPlugin, self).__init__()
        confspec = {
            "lockMicrophoneVolume": "boolean(default=False)",
            "microphoneVolume": "integer(default=50)"
        }
        config.conf.spec["audioManager"] = confspec
        self.audioNavigator = None
        self.playbackDeviceNavigator = PlaybackDeviceNavigator()
        self.recordingDeviceNavigator = RecordingDeviceNavigator()
        self.sessionNavigator = SessionNavigator()
        self.microphoneVolume = 0
        self.lockMicrophoneVolume = config.conf["audioManager"]["lockMicrophoneVolume"]
        self.microphoneVolume = config.conf["audioManager"]["microphoneVolume"] if self.lockMicrophoneVolume else 0
        self.timer = wx.Timer()
        self.timer.Bind(wx.EVT_TIMER, self.onTimer)
        self.timer.Start(100)
        self.nvdaOutputDeviceNavigator = NVDAOutputDeviceNavigator()

    def terminate(self):
        if self.lockMicrophoneVolume:
            config.conf["audioManager"]["microphoneVolume"] = self.microphoneVolume

    def _resetAllSessionDevice(self):
        manager = AudioManager()
        manager.initialize()
        manager.resetAllSessionDevice()
        manager.uninitialize()
        ui.message(_("Reset all playback and recording devices"))

    def _resetAllSessionVolume(self):
        manager = AudioManager()
        manager.initialize()
        sessionCount = manager.getSessionCount()
        for i in range(sessionCount):
            manager.setSessionVolume(i, 100)
            manager.setSessionMute(i, False)
        manager.uninitialize()
        ui.message(_("Reset volume and unmute all applications"))

    def onTimer(self, event):
        if not self.lockMicrophoneVolume:
            return
        try:
            manager = AudioManager()
            manager.initialize()
            currentRecordingDevice = manager.GetDefaultRecordingDevice()
            manager.setRecordingDeviceVolume(currentRecordingDevice, self.microphoneVolume)
            manager.uninitialize()
        except:
            pass

    def getMicrophoneVolume(self):
        manager = AudioManager()
        manager.initialize()
        currentRecordingDevice = manager.GetDefaultRecordingDevice()
        volume = manager.getRecordingDeviceVolume(currentRecordingDevice)
        manager.uninitialize()
        return volume

    @scriptHandler.script(
        category=CATEGORY_NAME,
        description=_("Next playback device"),
        gestures=["kb:control+windows+alt+numpad6", "kb(laptop):control+windows+alt+pagedown"]
    )
    def script_nextPlaybackDevice(self, gesture):
        self.playbackDeviceNavigator.next()
        self.audioNavigator = self.playbackDeviceNavigator

    @scriptHandler.script(
        category=CATEGORY_NAME,
        description=_("Previous playback device"),
        gestures=["kb:control+windows+alt+numpad4", "kb(laptop):control+windows+alt+pageup"]
    )
    def script_previousPlaybackDevice(self, gesture):
        self.playbackDeviceNavigator.previous()
        self.audioNavigator = self.playbackDeviceNavigator

    @scriptHandler.script(
        category=CATEGORY_NAME,
        description=_("Next recording device"),
        gestures=["kb:control+windows+alt+numpad3", "kb(laptop):control+windows+alt+end"]
    )
    def script_nextRecordingDevice(self, gesture):
        self.recordingDeviceNavigator.next()
        self.audioNavigator = self.recordingDeviceNavigator

    @scriptHandler.script(
        category=CATEGORY_NAME,
        description=_("Previous recording device"),
        gestures=["kb:control+windows+alt+numpad1","kb(laptop):control+windows+alt+home"]
    )
    def script_previousRecordingDevice(self, gesture):
        self.recordingDeviceNavigator.previous()
        self.audioNavigator = self.recordingDeviceNavigator

    @scriptHandler.script(
        category=CATEGORY_NAME,
        description=_("Next audio application"),
        gestures=["kb:control+windows+alt+numpad9", "kb(laptop):control+windows+alt+rightarrow"]
    )
    def script_nextSession(self, gesture):
        self.sessionNavigator.next()
        self.audioNavigator = self.sessionNavigator

    @scriptHandler.script(
        category=CATEGORY_NAME,
        description=_("Previous audio application"),
        gestures=["kb:control+windows+alt+numpad7", "kb(laptop):control+windows+alt+leftarrow"]
    )
    def script_previousSession(self, gesture):
        self.sessionNavigator.previous()
        self.audioNavigator = self.sessionNavigator

    @scriptHandler.script(
        category=CATEGORY_NAME,
        description=_("Increase volume"),
        gestures=["kb:control+windows+alt+numpad8", "kb(laptop):control+windows+alt+uparrow"]
    )
    def script_volumeUp(self, gesture):
        if self.audioNavigator is None:
            ui.message(_("Select a device or application first"))
            return
        self.audioNavigator.volumeUp()
        if isinstance(self.audioNavigator, RecordingDeviceNavigator):
            self.microphoneVolume = self.getMicrophoneVolume()
            if self.lockMicrophoneVolume:
                config.conf["audioManager"]["microphoneVolume"] = self.microphoneVolume

    @scriptHandler.script(
        category=CATEGORY_NAME,
        description=_("Decrease volume"),
        gestures=["kb:control+windows+alt+numpad2", "kb(laptop):control+windows+alt+downarrow"]
    )
    def script_volumeDown(self, gesture):
        if self.audioNavigator is None:
            ui.message(_("Select a device or application first"))
            return
        self.audioNavigator.volumeDown()
        if isinstance(self.audioNavigator, RecordingDeviceNavigator):
            self.microphoneVolume = self.getMicrophoneVolume()
            if self.lockMicrophoneVolume:
                config.conf["audioManager"]["microphoneVolume"] = self.microphoneVolume

    @scriptHandler.script(
        category=CATEGORY_NAME,
        description=_("Toggle mute"),
        gestures=["kb:control+windows+alt+numpad5", "kb(laptop):control+windows+alt+space"]
    )
    def script_mute(self, gesture):
        if self.audioNavigator is not None:
            self.audioNavigator.mute()
        else:
            ui.message(_("Select a device or application first"))

    @scriptHandler.script(
        category=CATEGORY_NAME,
        description=_("Set as default device"),
        gestures=["kb:control+windows+alt+numpadenter", "kb(laptop):control+windows+alt+enter"]
    )
    def script_asDefault(self, gesture):
        if isinstance(self.audioNavigator, PlaybackDeviceNavigator) or isinstance(self.audioNavigator, RecordingDeviceNavigator):
            self.audioNavigator.asDefault()
        else:
            ui.message(_("Select a playback or recording device first"))

    @scriptHandler.script(
        category=CATEGORY_NAME,
        description=_("Next playback device for app"),
        gestures=["kb:control+windows+alt+numpadmultiply", "kb(laptop):control+windows+alt+]"]
    )
    def script_nextPlaybackDeviceOfApplication(self, gesture):
        if isinstance(self.audioNavigator, SessionNavigator):
            self.audioNavigator.nextPlaybackDevice()
        else:
            ui.message(_("Select an audio application first"))

    @scriptHandler.script(
        category=CATEGORY_NAME,
        description=_("Previous playback device for app"),
        gestures=["kb:shift+control+windows+alt+numpadmultiply", "kb(laptop):shift+control+windows+alt+]"]
    )
    def script_previousPlaybackDeviceOfApplication(self, gesture):
        if isinstance(self.audioNavigator, SessionNavigator):
            self.audioNavigator.previousPlaybackDevice()
        else:
            ui.message(_("Select an audio application first"))

    @scriptHandler.script(
        category=CATEGORY_NAME,
        description=_("Previous recording device for app"),
        gestures=["kb:shift+control+windows+alt+numpaddivide", "kb(laptop):shift+control+windows+alt+["]
    )
    def script_previousRecordingDeviceOfApplication(self, gesture):
        if isinstance(self.audioNavigator, SessionNavigator):
            self.audioNavigator.previousRecordingDevice()
        else:
            ui.message(_("Select an audio application first"))

    @scriptHandler.script(
        category=CATEGORY_NAME,
        description=_("Next recording device for app"),
        gestures=["kb:control+windows+alt+numpaddivide", "kb(laptop):control+windows+alt+["]
    )
    def script_nextRecordingDeviceOfApplication(self, gesture):
        if isinstance(self.audioNavigator, SessionNavigator):
            self.audioNavigator.nextRecordingDevice()
        else:
            ui.message(_("Select an audio application first"))

    @scriptHandler.script(
        category=CATEGORY_NAME,
        description=_("Reset devices and volume"),
        gestures=["kb:control+windows+alt+numpadminus", "kb(laptop):control+windows+alt+backspace"]
    )
    def script_resetDefault(self, gesture):
        repeatCount = scriptHandler.getLastScriptRepeatCount()
        if repeatCount == 0:
            self._resetAllSessionDevice()
        elif repeatCount == 1:
            self._resetAllSessionVolume()

    @scriptHandler.script(
        category=CATEGORY_NAME,
        description=_("Toggle microphone mute"),
        gestures=["kb:shift+control+windows+alt+numpad2" ,"kb(laptop):shift+control+windows+alt+downarrow"]
    )
    def script_microphoneSwitch(self, gesture):
        manager = AudioManager()
        manager.initialize()
        currentRecordingDevice = manager.GetDefaultRecordingDevice()
        muteStatus = manager.getRecordingDeviceMute(currentRecordingDevice)
        muteStatus = not muteStatus
        manager.setRecordingDeviceMute(currentRecordingDevice, muteStatus)
        muteStatus = manager.getRecordingDeviceMute(currentRecordingDevice)
        message = _("Microphone is off") if muteStatus else _("Microphone is on")
        ui.message(message)
        manager.uninitialize()

    @scriptHandler.script(
        category=CATEGORY_NAME,
        description=_("Toggle sound mute"),
        gestures=["kb:shift+control+windows+alt+numpad5", "kb(laptop):shift+control+windows+alt+uparrow"]
    )
    def script_soundSwitch(self, gesture):
        manager = AudioManager()
        manager.initialize()
        currentPlaybackDevice = manager.GetDefaultPlaybackDevice()
        muteStatus = manager.getPlaybackDeviceMute(currentPlaybackDevice)
        muteStatus = not muteStatus
        manager.setPlaybackDeviceMute(currentPlaybackDevice, muteStatus)
        muteStatus = manager.getPlaybackDeviceMute(currentPlaybackDevice)
        message = _("Sound is off") if muteStatus else _("Sound is on")
        ui.message(message)
        manager.uninitialize()

    @scriptHandler.script(
        category=CATEGORY_NAME,
        description=_("Toggle window mute"),
        gestures=["kb:shift+control+windows+alt+numpad8", "kb(laptop):shift+control+windows+alt+space"]
    )
    def script_muteWindow(self, gesture):
        manager = AudioManager()
        manager.initialize()
        mute = manager.getWindowMute()
        mute = not mute
        manager.setWindowMute(mute)
        mute = manager.getWindowMute()
        message = _("Window is muted") if mute else _("Window is unmuted")
        ui.message(message)
        manager.uninitialize()

    @scriptHandler.script(
        category=CATEGORY_NAME,
        description=_("Toggle microphone volume lock"),
        gestures=["kb:control+windows+alt+numpaddelete", "kb(laptop):control+windows+alt+delete"]
    )
    def script_lockMicrophoneVolume(self, gesture):
        self.microphoneVolume = self.getMicrophoneVolume()
        self.lockMicrophoneVolume = not self.lockMicrophoneVolume
        if self.lockMicrophoneVolume:
            config.conf["audioManager"]["microphoneVolume"] = self.microphoneVolume
        message = _("Microphone volume locked") if self.lockMicrophoneVolume else _("Microphone volume unlocked")
        config.conf["audioManager"]["lockMicrophoneVolume"] = self.lockMicrophoneVolume
        ui.message(message)

    @scriptHandler.script(
        category=CATEGORY_NAME,
        description=_("Next NVDA output device"),
        gestures=["kb:control+windows+alt+numpadplus", "kb(laptop):control+windows+alt+\\"]
    )
    def script_nextNVDAOutputDevice(self, gesture):
        self.nvdaOutputDeviceNavigator.next()

    @scriptHandler.script(
        category=CATEGORY_NAME,
        description=_("Previous NVDA output device"),
        gestures=["kb:shift+control+windows+alt+numpadplus", "kb(laptop):shift+control+windows+alt+\\"]
    )
    def script_previousNVDAOutputDevice(self, gesture):
        self.nvdaOutputDeviceNavigator.previous()

    @scriptHandler.script(
        category=CATEGORY_NAME,
        description=_("Open Sound Control Panel"),
        gestures=["kb:control+windows+alt+o"]
    )
    def script_openSoundControlPanel(self, gesture):
        try:
            os.system("control mmsys.cpl")
        except Exception as e:
            ui.message(_("Failed to open Sound Control Panel"))