# pip install wxpython accessible-output2 pyperclip

"""
Accessible Cron Line Generator (wxPython GUI)

Features:
- Each cron field (Minute, Hour, etc.) is grouped with wx.StaticBox
- Alt+hotkeys for accessibility
- Supports Any Value (*) and Every N (*/N)
- Live preview of cron line
- Copy to clipboard / Reset / Exit
"""

import wx
import pyperclip

FIELDS = [
    ("&Minute", 0, 59, "minute"),
    ("&Hour", 0, 23, "hour"),
    ("Day of &Month", 1, 31, "day_of_month"),
    ("&Month", 1, 12, "month"),
    ("Day of &Week", 0, 7, "day_of_week"),
]


class CronGeneratorFrame(wx.Frame):
    def __init__(self):
        super().__init__(None, title="Cron Line Generator", size=(560, 600))
        panel = wx.Panel(self)
        main_sizer = wx.BoxSizer(wx.VERTICAL)

        # Header
        header = wx.StaticText(panel, label="Cron Line Generator")
        header_font = header.GetFont()
        header_font.PointSize += 2
        header_font = header_font.Bold()
        header.SetFont(header_font)
        main_sizer.Add(header, 0, wx.ALL | wx.CENTER, 10)

        self.field_controls = []

        # Create grouped sections for each field
        for label, low, high, key in FIELDS:
            box = wx.StaticBox(panel, label=label)
            box_sizer = wx.StaticBoxSizer(box, wx.HORIZONTAL)

            spin = wx.SpinCtrl(panel, min=low, max=high)
            spin.Disable()

            any_check = wx.CheckBox(panel, label="Any (*)")
            any_check.SetValue(True)

            every_check = wx.CheckBox(panel, label="Every N (*/)")
            every_spin = wx.SpinCtrl(panel, min=1, max=60, initial=5)
            every_spin.Disable()

            # Bind handlers
            any_check.Bind(wx.EVT_CHECKBOX,
                           lambda e, s=spin, ec=every_check, es=every_spin: self.on_any(e, s, ec, es))
            every_check.Bind(wx.EVT_CHECKBOX,
                             lambda e, s=spin, ac=any_check, es=every_spin: self.on_every(e, s, ac, es))
            spin.Bind(wx.EVT_SPINCTRL, self.update_preview)
            every_spin.Bind(wx.EVT_SPINCTRL, self.update_preview)

            # Add to group box
            inner_sizer = wx.BoxSizer(wx.HORIZONTAL)
            inner_sizer.Add(spin, 0, wx.ALL | wx.ALIGN_CENTER_VERTICAL, 5)
            inner_sizer.Add(any_check, 0, wx.ALL | wx.ALIGN_CENTER_VERTICAL, 5)
            inner_sizer.Add(every_check, 0, wx.ALL | wx.ALIGN_CENTER_VERTICAL, 5)
            inner_sizer.Add(every_spin, 0, wx.ALL | wx.ALIGN_CENTER_VERTICAL, 5)

            box_sizer.Add(inner_sizer, 1, wx.EXPAND)
            main_sizer.Add(box_sizer, 0, wx.ALL | wx.EXPAND, 5)

            self.field_controls.append((spin, any_check, every_check, every_spin))

        # Command section
        cmd_box = wx.StaticBox(panel, label="Command to run (&C)")
        cmd_sizer = wx.StaticBoxSizer(cmd_box, wx.VERTICAL)
        self.cmd_text = wx.TextCtrl(panel)
        self.cmd_text.Bind(wx.EVT_TEXT, self.update_preview)
        cmd_sizer.Add(self.cmd_text, 0, wx.ALL | wx.EXPAND, 5)
        main_sizer.Add(cmd_sizer, 0, wx.ALL | wx.EXPAND, 10)

        # Live preview section
        preview_box = wx.StaticBox(panel, label="Generated cron line")
        preview_sizer = wx.StaticBoxSizer(preview_box, wx.VERTICAL)
        self.preview_ctrl = wx.TextCtrl(panel,
                                        style=wx.TE_MULTILINE | wx.TE_READONLY | wx.TE_RICH2,
                                        size=(-1, 70))
        self.preview_ctrl.SetBackgroundColour(wx.Colour(245, 245, 245))
        preview_sizer.Add(self.preview_ctrl, 1, wx.ALL | wx.EXPAND, 5)
        main_sizer.Add(preview_sizer, 0, wx.ALL | wx.EXPAND, 10)

        # Buttons
        btn_sizer = wx.BoxSizer(wx.HORIZONTAL)
        copy_btn = wx.Button(panel, label="Copy to Clipboard (&O)")
        new_btn = wx.Button(panel, label="New Line (&N)")
        quit_btn = wx.Button(panel, label="Exit (&X)")

        copy_btn.Bind(wx.EVT_BUTTON, self.on_copy)
        new_btn.Bind(wx.EVT_BUTTON, self.on_new)
        quit_btn.Bind(wx.EVT_BUTTON, lambda e: self.Close())

        btn_sizer.Add(copy_btn, 1, wx.ALL, 5)
        btn_sizer.Add(new_btn, 1, wx.ALL, 5)
        btn_sizer.Add(quit_btn, 1, wx.ALL, 5)
        main_sizer.Add(btn_sizer, 0, wx.ALL | wx.EXPAND, 10)

        panel.SetSizer(main_sizer)
        self.update_preview(None)

        self.Centre()
        self.Show()

    # --- Control logic ---
    def on_any(self, event, spin, every_check, every_spin):
        check = event.GetEventObject()
        if check.IsChecked():
            spin.Disable()
            every_check.SetValue(False)
            every_spin.Disable()
        else:
            spin.Enable()
        self.update_preview(None)

    def on_every(self, event, spin, any_check, every_spin):
        check = event.GetEventObject()
        if check.IsChecked():
            every_spin.Enable()
            any_check.SetValue(False)
            spin.Disable()
        else:
            every_spin.Disable()
            spin.Enable()
        self.update_preview(None)

    def get_field_values(self):
        values = []
        for spin, any_check, every_check, every_spin in self.field_controls:
            if any_check.GetValue():
                values.append("*")
            elif every_check.GetValue():
                values.append(f"*/{every_spin.GetValue()}")
            else:
                values.append(str(spin.GetValue()))
        return values

    def update_preview(self, event):
        values = self.get_field_values()
        cmd = self.cmd_text.GetValue().strip()
        cron_line = " ".join(values) + (" " + cmd if cmd else "")
        self.preview_ctrl.SetValue(cron_line)

    def on_copy(self, event):
        cron_line = self.preview_ctrl.GetValue().strip()
        if cron_line:
            try:
                pyperclip.copy(cron_line)
                wx.MessageBox("Cron line copied to clipboard.", "Copied",
                              wx.OK | wx.ICON_INFORMATION)
            except pyperclip.PyperclipException:
                wx.MessageBox("Clipboard not accessible.", "Error",
                              wx.OK | wx.ICON_ERROR)

    def on_new(self, event):
        for spin, any_check, every_check, every_spin in self.field_controls:
            any_check.SetValue(True)
            every_check.SetValue(False)
            spin.SetValue(spin.GetMin())
            spin.Disable()
            every_spin.Disable()
        self.cmd_text.SetValue("")
        self.update_preview(None)


class CronApp(wx.App):
    def OnInit(self):
        frame = CronGeneratorFrame()
        frame.Show()
        return True


if __name__ == "__main__":
    app = CronApp(False)
    app.MainLoop()
