# pip install wxpython mistune

import wx
import mistune

class MarkdownConverter(wx.Frame):
    def __init__(self, parent, title):
        super(MarkdownConverter, self).__init__(parent, title=title, size=(450, 300))

        # Panel
        panel = wx.Panel(self)

        # Layout
        vbox = wx.BoxSizer(wx.VERTICAL)

        # Title Input
        title_label = wx.StaticText(panel, label="HTML Page Title:")
        self.title_input = wx.TextCtrl(panel)
        vbox.Add(title_label, flag=wx.EXPAND | wx.LEFT | wx.RIGHT | wx.TOP, border=10)
        vbox.Add(self.title_input, flag=wx.EXPAND | wx.LEFT | wx.RIGHT | wx.TOP, border=10)

        # Markdown Filename Input
        file_label = wx.StaticText(panel, label="Markdown Filename (without .md):")
        self.file_input = wx.TextCtrl(panel)
        vbox.Add(file_label, flag=wx.EXPAND | wx.LEFT | wx.RIGHT | wx.TOP, border=10)
        vbox.Add(self.file_input, flag=wx.EXPAND | wx.LEFT | wx.RIGHT | wx.TOP, border=10)

        # Convert Button
        convert_button = wx.Button(panel, label="Convert to HTML")
        convert_button.Bind(wx.EVT_BUTTON, self.on_convert)
        vbox.Add(convert_button, flag=wx.ALIGN_CENTER | wx.ALL, border=10)

        # Status message
        self.status_msg = wx.StaticText(panel, label="")
        vbox.Add(self.status_msg, flag=wx.ALIGN_CENTER | wx.ALL, border=10)

        panel.SetSizer(vbox)

    def on_convert(self, event):
        title = self.title_input.GetValue().strip()
        markdown_filename = self.file_input.GetValue().strip()

        if not title or not markdown_filename:
            self.status_msg.SetLabel("Please fill in both fields.")
            return

        # Read Markdown file content
        markdown_file = markdown_filename + '.md'
        try:
            with open(markdown_file, 'r', encoding='utf-8') as md_file:
                markdown_content = md_file.read()
        except FileNotFoundError:
            self.status_msg.SetLabel(f"Error: The file {markdown_file} does not exist.")
            return

        # Use Mistune to parse the markdown content
        markdown = mistune.create_markdown()
        html_content = markdown(markdown_content)

        # HTML template with basic styling
        html_template = f'''
        <!DOCTYPE html>
        <html lang="en">
        <head>
            <meta charset="UTF-8">
            <meta name="viewport" content="width=device-width, initial-scale=1.0">
            <title>{title}</title>
            <style>
                body {{
                    font-family: Arial, sans-serif;
                    line-height: 1.6;
                    max-width: 800px;
                    margin: 20px auto;
                    padding: 0 10px;
                }}
                h1, h2, h3, h4, h5, h6 {{
                    color: #333;
                }}
                p {{
                    color: #555;
                }}
                pre {{
                    background-color: #f4f4f4;
                    padding: 10px;
                    border-radius: 5px;
                    overflow-x: auto;
                }}
                code {{
                    background-color: #f4f4f4;
                    padding: 2px 4px;
                    border-radius: 3px;
                }}
                a {{
                    color: #1e90ff;
                }}
                blockquote {{
                    border-left: 3px solid #ddd;
                    padding-left: 10px;
                    color: #777;
                    font-style: italic;
                }}
                ul, ol {{
                    margin-left: 20px;
                }}
            </style>
        </head>
        <body>
            {html_content}
        </body>
        </html>
        '''

        # Write the generated HTML to a file
        html_filename = markdown_filename + '.html'
        with open(html_filename, 'w', encoding='utf-8') as html_file:
            html_file.write(html_template)

        self.status_msg.SetLabel(f"HTML file '{html_filename}' has been generated successfully.")

def main():
    app = wx.App()
    frame = MarkdownConverter(None, title="Markdown to HTML Converter")
    frame.Show()
    app.MainLoop()

if __name__ == "__main__":
    main()