# pip install wxpython pyprowl tzlocal

import wx
from pyprowl import Prowl
from datetime import datetime
from tzlocal import get_localzone  # This library is used to get the local timezone

def get_api_key():
    file_path = "apikey.txt"
    try:
        with open(file_path, "r") as f:
            api_key = f.read().strip()
            if api_key:
                return api_key
    except FileNotFoundError:
        pass

    # Prompting user for the API key
    api_key = wx.GetTextFromUser("Please enter your API key:", "Enter API Key").strip()

    # If the user cancels or provides no key, return None
    if not api_key:
        return None

    with open(file_path, "w") as f:
        f.write(api_key)

    return api_key

class MyApp(wx.App):
    def OnInit(self):
        api_key = get_api_key()
        if api_key is None:
            wx.MessageBox("No API key provided. Exiting application.", "Error", wx.OK | wx.ICON_ERROR)
            return False  # Exit the application

        self.frame = MainFrame(None, title="Prowl Notification", api_key=api_key)
        self.frame.Show()
        return True

class MainFrame(wx.Frame):
    def __init__(self, parent, title, api_key):
        super(MainFrame, self).__init__(parent, title=title, size=(400, 600))  # Increased height for more space
        self.api_key = api_key

        panel = wx.Panel(self)
        vbox = wx.BoxSizer(wx.VERTICAL)

        # Application Name
        app_name_lbl = wx.StaticText(panel, label="&Application Name:")  # Alt+A for access
        self.app_name_txt = wx.TextCtrl(panel)
        self.app_name_txt.SetHint("Enter application name")
        vbox.Add(app_name_lbl, flag=wx.EXPAND | wx.ALL, border=5)
        vbox.Add(self.app_name_txt, flag=wx.EXPAND | wx.ALL, border=5)

        # Event Name
        event_lbl = wx.StaticText(panel, label="&Event Name:")  # Alt+E for access
        self.event_txt = wx.TextCtrl(panel)
        self.event_txt.SetHint("Enter event name")
        vbox.Add(event_lbl, flag=wx.EXPAND | wx.ALL, border=5)
        vbox.Add(self.event_txt, flag=wx.EXPAND | wx.ALL, border=5)

        # Event Description
        description_lbl = wx.StaticText(panel, label="&Event Description:")  # Alt+D for access
        self.description_txt = wx.TextCtrl(panel, style=wx.TE_MULTILINE)
        self.description_txt.SetHint("Enter event description")
        vbox.Add(description_lbl, flag=wx.EXPAND | wx.ALL, border=5)
        vbox.Add(self.description_txt, proportion=1, flag=wx.EXPAND | wx.ALL, border=5)

        # Priority
        priority_lbl = wx.StaticText(panel, label="P&riority:")  # Alt+R for access
        self.priority_choice = wx.Choice(panel, choices=["Emergency", "High", "Normal", "Moderate", "Very Low"])
        self.priority_choice.SetSelection(2)
        vbox.Add(priority_lbl, flag=wx.EXPAND | wx.ALL, border=5)
        vbox.Add(self.priority_choice, flag=wx.EXPAND | wx.ALL, border=5)

        # Send Notification Button
        send_button = wx.Button(panel, label="&Send Notification")  # Alt+S for execution
        send_button.Bind(wx.EVT_BUTTON, self.send_notification)
        vbox.Add(send_button, flag=wx.ALIGN_CENTER | wx.TOP | wx.BOTTOM, border=10)

        # Verify API Key Button
        verify_button = wx.Button(panel, label="&Verify API Key")  # Alt+V for execution
        verify_button.Bind(wx.EVT_BUTTON, self.verify_api_key)
        vbox.Add(verify_button, flag=wx.ALIGN_CENTER | wx.TOP | wx.BOTTOM, border=10)

        # Label for the output area
        output_lbl = wx.StaticText(panel, label="O&utput:")  # Alt+U for focus
        vbox.Add(output_lbl, flag=wx.EXPAND | wx.ALL, border=5)

        # Output read-only text box for response
        self.output_txt = wx.TextCtrl(panel, style=wx.TE_MULTILINE | wx.TE_READONLY)
        vbox.Add(self.output_txt, proportion=1, flag=wx.EXPAND | wx.ALL, border=5)

        panel.SetSizer(vbox)

        # Connect focus to associated inputs when Alt key combinations are used
        app_name_lbl.Bind(wx.EVT_LEFT_UP, lambda event: self.app_name_txt.SetFocus())
        event_lbl.Bind(wx.EVT_LEFT_UP, lambda event: self.event_txt.SetFocus())
        description_lbl.Bind(wx.EVT_LEFT_UP, lambda event: self.description_txt.SetFocus())
        priority_lbl.Bind(wx.EVT_LEFT_UP, lambda event: self.priority_choice.SetFocus())

    def send_notification(self, event):
        application = self.app_name_txt.GetValue().strip()
        event_description = self.event_txt.GetValue().strip()
        description = self.description_txt.GetValue().strip()

        # Clear previous output before sending a new notification
        self.output_txt.Clear()

        priority_dict = {
            0: 2,  # Emergency
            1: 1,  # High
            2: 0,  # Normal
            3: -1, # Moderate
            4: -2  # Very Low
        }
        priority = priority_dict[self.priority_choice.GetSelection()]

        try:
            prowl = Prowl(self.api_key)
            response = prowl.add(
                appName=application,
                event=event_description,
                description=description,
                priority=priority
            )

            self.display_response(response, "Notification sent successfully!")
            wx.MessageBox("Notification sent successfully! See the output box for more details.", 
                          "Success", wx.OK | wx.ICON_INFORMATION)
        except Exception as e:
            wx.MessageBox(f"Error sending notification: {str(e)}", "Error", wx.OK | wx.ICON_ERROR)

    def verify_api_key(self, event):
        # Clear previous output before verifying the API key
        self.output_txt.Clear()

        try:
            prowl = Prowl(self.api_key)
            response = prowl.verify()

            self.display_response(response, "API Key verified successfully!")
            wx.MessageBox("API Key verified successfully! See the output box for more details.", 
                          "Success", wx.OK | wx.ICON_INFORMATION)
        except Exception as e:
            wx.MessageBox(f"Error verifying API key: {str(e)}", "Error", wx.OK | wx.ICON_ERROR)

    def display_response(self, response, success_message):
        # Constructing a human-readable output from the response dictionary
        if isinstance(response, dict):
            # Convert resetdate from Unix timestamp to a formatted local date
            reset_timestamp = int(response.get('resetdate', 0))
            local_timezone = get_localzone()
            reset_datetime = datetime.fromtimestamp(reset_timestamp, local_timezone)
            reset_date_str = reset_datetime.strftime("%A, %B %d, %Y at %I:%M:%S %p")

            # Calculate the time remaining until reset
            current_time = datetime.now(local_timezone)
            time_remaining_sec = (reset_datetime - current_time).total_seconds()
            time_remaining_min = int(time_remaining_sec // 60)

            output_message = (
                f"{success_message}\n\nResults:\n"
                f"API Calls Remaining: {response.get('remaining', 'N/A')}\n"
                f"Your API calls will reset in {time_remaining_min} minutes (on {reset_date_str})."
            )
            self.output_txt.SetValue(output_message)

if __name__ == "__main__":
    app = MyApp()
    app.MainLoop()