# pip install wxpython pyperclip

import wx
import string
import secrets
import configparser
import os
import pyperclip

CONFIG_FILE = "config.ini"
MAX_PASSWORD_LENGTH = 1024  # Maximum allowed password length
MIN_PASSWORD_LENGTH = 1     # Minimum allowed password length

def load_config():
    """Loads the saved configuration from an INI file or defaults."""
    config = configparser.ConfigParser()
    settings = {
        'length': 12,
        'use_upper': True,
        'use_lower': True,
        'use_digits': True,
        'use_punctuation': True
    }
    if os.path.exists(CONFIG_FILE) and os.path.getsize(CONFIG_FILE) > 0:
        config.read(CONFIG_FILE)
        try:
            settings['length'] = config.getint('Settings', 'length')
            settings['use_upper'] = config.getboolean('Settings', 'use_upper')
            settings['use_lower'] = config.getboolean('Settings', 'use_lower')
            settings['use_digits'] = config.getboolean('Settings', 'use_digits')
            settings['use_punctuation'] = config.getboolean('Settings', 'use_punctuation')
            # Validate loaded length
            if not (MIN_PASSWORD_LENGTH <= settings['length'] <= MAX_PASSWORD_LENGTH):
                settings['length'] = MIN_PASSWORD_LENGTH  # Default to minimum if out of bounds
                wx.LogError("Invalid password length in config file. Resetting to minimum value.")
        except (configparser.NoOptionError, configparser.NoSectionError):
            save_config(settings)
    else:
        save_config(settings)
    return settings

def save_config(settings):
    """Saves the current settings to an INI file."""
    config = configparser.ConfigParser()
    config['Settings'] = {key: str(value) for key, value in settings.items()}
    with open(CONFIG_FILE, 'w') as configfile:
        config.write(configfile, space_around_delimiters=False)

def generate_password(length, use_upper, use_lower, use_digits, use_punctuation):
    """Generates a random password that includes at least one character from each selected type."""
    pools = {}
    if use_upper:
        pools['upper'] = string.ascii_uppercase
    if use_lower:
        pools['lower'] = string.ascii_lowercase
    if use_digits:
        pools['digits'] = string.digits
    if use_punctuation:
        pools['punctuation'] = string.punctuation

    if not pools:
        raise ValueError("No character sets selected. Enable at least one character type.")

    if length < len(pools):
        raise ValueError("Password length is too short to include at least one character from each selected set.")

    # Start by selecting one character from each enabled set.
    password_chars = [secrets.choice(pool) for pool in pools.values()]

    # Build a combined pool for the remaining characters.
    combined_pool = ''.join(pools.values())
    password_chars.extend(secrets.choice(combined_pool) for _ in range(length - len(password_chars)))

    # Shuffle so the ensured characters are not in a predictable position.
    secrets.SystemRandom().shuffle(password_chars)

    return ''.join(password_chars)

class PasswordGeneratorApp(wx.Frame):
    def __init__(self, parent, title):
        super(PasswordGeneratorApp, self).__init__(parent, title=title, size=(400, 350))

        panel = wx.Panel(self)
        settings = load_config()

        vbox = wx.BoxSizer(wx.VERTICAL)

        # Password length spin button
        length_label = wx.StaticText(panel, label="Password &Length:")
        vbox.Add(length_label, flag=wx.EXPAND|wx.LEFT|wx.RIGHT|wx.TOP, border=10)
        self.length_spin = wx.SpinCtrl(panel, value=str(settings['length']), min=MIN_PASSWORD_LENGTH, max=MAX_PASSWORD_LENGTH)
        vbox.Add(self.length_spin, flag=wx.EXPAND|wx.LEFT|wx.RIGHT|wx.BOTTOM, border=10)

        # Use upper case
        self.use_upper_checkbox = wx.CheckBox(panel, label="Include &Uppercase Letters")
        self.use_upper_checkbox.SetValue(settings['use_upper'])
        vbox.Add(self.use_upper_checkbox, flag=wx.EXPAND|wx.LEFT|wx.RIGHT, border=10)

        # Use lower case
        self.use_lower_checkbox = wx.CheckBox(panel, label="Include &Lowercase Letters")
        self.use_lower_checkbox.SetValue(settings['use_lower'])
        vbox.Add(self.use_lower_checkbox, flag=wx.EXPAND|wx.LEFT|wx.RIGHT, border=10)

        # Use digits
        self.use_digits_checkbox = wx.CheckBox(panel, label="Include &Digits")
        self.use_digits_checkbox.SetValue(settings['use_digits'])
        vbox.Add(self.use_digits_checkbox, flag=wx.EXPAND|wx.LEFT|wx.RIGHT, border=10)

        # Use punctuation
        self.use_punctuation_checkbox = wx.CheckBox(panel, label="Include &Punctuation")
        self.use_punctuation_checkbox.SetValue(settings['use_punctuation'])
        vbox.Add(self.use_punctuation_checkbox, flag=wx.EXPAND|wx.LEFT|wx.RIGHT, border=10)

        # Generate button
        self.generate_button = wx.Button(panel, label="&Generate and Copy Password")
        vbox.Add(self.generate_button, flag=wx.ALIGN_CENTER|wx.TOP|wx.BOTTOM, border=20)
        self.generate_button.Bind(wx.EVT_BUTTON, self.on_generate_password)

        panel.SetSizer(vbox)

    def on_generate_password(self, event):
        length = self.length_spin.GetValue()
        use_upper = self.use_upper_checkbox.GetValue()
        use_lower = self.use_lower_checkbox.GetValue()
        use_digits = self.use_digits_checkbox.GetValue()
        use_punctuation = self.use_punctuation_checkbox.GetValue()

        try:
            password = generate_password(length, use_upper, use_lower, use_digits, use_punctuation)
            pyperclip.copy(password)
            wx.MessageBox("Generated password has been copied to your clipboard!", "Success", wx.OK | wx.ICON_INFORMATION)
        except ValueError as e:
            wx.MessageBox(str(e), "Error", wx.OK | wx.ICON_ERROR)

        settings = {
            'length': length,
            'use_upper': use_upper,
            'use_lower': use_lower,
            'use_digits': use_digits,
            'use_punctuation': use_punctuation
        }
        save_config(settings)

if __name__ == "__main__":
    app = wx.App()
    frame = PasswordGeneratorApp(None, title="PWGen - Password Generator")
    frame.Show()
    app.MainLoop()