import os
import subprocess
import platform

def prompt_directory():
    while True:
        try:
            directory = input("Enter the directory to search (leave blank for current directory): ").strip()
            if not directory:
                directory = os.getcwd()
            if not os.path.isdir(directory):
                print("Error: That is not a valid directory. Please try again.")
                continue
            return directory
        except Exception as e:
            print(f"Unexpected error: {e}")

def prompt_search_string():
    while True:
        search_string = input("Enter the string to search for: ").strip()
        if search_string:
            return search_string
        else:
            print("Search string cannot be empty. Please try again.")

def prompt_case_sensitivity():
    while True:
        choice = input("Should the search be case sensitive? (y/N): ").strip().lower()
        if choice in ('y', 'n', ''):
            return choice == 'y'
        else:
            print("Please enter 'y', 'n', or just press Enter for default (No).")

def search_files(base_dir, search_string, case_sensitive):
    matches = {}
    file_counter = 1

    for root, _, files in os.walk(base_dir):
        for name in files:
            full_path = os.path.join(root, name)
            try:
                with open(full_path, 'r', encoding='utf-8', errors='ignore') as f:
                    contents = f.read()
                    if (search_string in contents) if case_sensitive else (search_string.lower() in contents.lower()):
                        matches[file_counter] = full_path
                        print(f"{file_counter}: {os.path.relpath(full_path, base_dir)}")
                        file_counter += 1
            except Exception as e:
                print(f"Could not read {full_path}: {e}")

    return matches

def open_file(path):
    try:
        system = platform.system()
        if system == 'Windows':
            # Open with Notepad explicitly
            subprocess.run(['notepad.exe', path], check=False)
        elif system == 'Darwin':  # macOS
            subprocess.run(['open', path], check=False)
        else:  # Linux and others
            subprocess.run(['xdg-open', path], check=False)
    except Exception as e:
        print(f"Failed to open file: {e}")

def main():
    base_dir = prompt_directory()
    search_string = prompt_search_string()
    case_sensitive = prompt_case_sensitivity()

    print("\nSearching...\n")
    matches = search_files(base_dir, search_string, case_sensitive)

    if not matches:
        print("No matches found.")
        return

    while True:
        try:
            choice = input("\nEnter the number of the file to open, or 0 to cancel: ").strip()
            if not choice.isdigit():
                raise ValueError("Input must be a number.")
            choice = int(choice)
            if choice == 0:
                print("Canceled.")
                return
            elif choice in matches:
                print(f"Opening {matches[choice]}...")
                open_file(matches[choice])
                return
            else:
                print("Invalid selection. Please choose a number from the list.")
        except ValueError as ve:
            print(f"Error: {ve}")
        except Exception as e:
            print(f"Unexpected error: {e}")

if __name__ == "__main__":
    main()