import os
import sys
import math
import io

import globalPluginHandler
import tones
import addonHandler
import nvwave

# Initialize translation and metadata
addonHandler.initTranslation()
addonName = addonHandler.getCodeAddon().manifest["name"]

def log(msg):
    print(f"{addonName}: {msg}")

log("Loading Custom Tones (nvwave + pydub backend)")

# Load plugin-local libs (pydub)
addon_dir = os.path.dirname(__file__)
lib_path = os.path.join(addon_dir, "lib")
sys.path.insert(0, lib_path)

try:
    from pydub import AudioSegment
    pydub_loaded = True
    log("pydub available")
except Exception as e:
    pydub_loaded = False
    log(f"Unable to import pydub: {e}")

# Base tone initialization
BASE_WAV_PATH = os.path.join(addon_dir, "sounds", "base_tone.wav")
BASE_WAV_FREQ = 440  # reference for pitch shifting
BASE = None

if pydub_loaded and os.path.exists(BASE_WAV_PATH):
    try:
        BASE = AudioSegment.from_file(BASE_WAV_PATH)
        log("Base tone loaded")
    except Exception as e:
        log(f"Error reading base sound: {e}")
else:
    if not pydub_loaded:
        log("pydub missing")
    if not os.path.exists(BASE_WAV_PATH):
        log("base_tone.wav missing")

# preserve original tone behavior
original_beep = tones.beep


def generate_beep(frequency, duration_ms, pan=0.0):
    """
    Build a custom tone from base_tone.wav:

    - pitch shifts to requested frequency (frame_rate trick)
    - loops audio to requested duration
    - applies stereo/pan
    - fade edges
    - returns a BytesIO wav file

    duration_ms: integer milliseconds
    pan: -1.0 = left, 0 centered, +1.0 right
    """

    if BASE is None:
        return None

    try:
        duration_ms = max(1, int(duration_ms))
        freq = max(1, float(frequency))

        # pitch shift
        semitones = 12 * math.log2(freq / BASE_WAV_FREQ)
        pitched = BASE._spawn(
            BASE.raw_data,
            overrides={
                "frame_rate": int(BASE.frame_rate * (2 ** (semitones / 12.0)))
            },
        ).set_frame_rate(BASE.frame_rate)

        seg = pitched

        # repeat audio until duration satisfied
        if len(seg) < duration_ms:
            source = seg
            while len(seg) < duration_ms:
                seg += source
        seg = seg[:duration_ms]

        # stereo + pan
        seg = seg.set_channels(2)
        seg = seg.pan(pan)

        # fade edges to avoid click
        fade_ms = 1
        if len(seg) >= fade_ms * 2:
            seg = seg.fade_in(fade_ms).fade_out(fade_ms)

        buf = io.BytesIO()
        seg.export(buf, format="wav")
        buf.seek(0)
        return buf

    except Exception as e:
        log(f"generate_beep failed: {e}")
        return None


def patched_beep(frequency, duration, *args, **kwargs):
    """
    Wrap tones.beep, preserve NVDA args:

    tones.beep signature may include:
        frequency, duration, leftRight=panValue,
        isSpeechBeepCommand=True, etc.

    We accept all kwargs.
    """

    try:
        # default pan if not supplied
        pan = 0.0
        if "leftRight" in kwargs:
            # NVDA uses -1.0 left +1.0 right
            try:
                pan = float(kwargs["leftRight"])
            except:
                pan = 0.0

        buf = generate_beep(frequency, duration, pan)

        if buf is None:
            return original_beep(frequency, duration, *args, **kwargs)

        # play synchronous to preserve NVDA behavior
        nvwave.playWaveFile(buf, False)

    except Exception as e:
        log(f"fallback to original beep: {e}")
        return original_beep(frequency, duration, *args, **kwargs)


class GlobalPlugin(globalPluginHandler.GlobalPlugin):
    def __init__(self):
        super(GlobalPlugin, self).__init__()

        if BASE and pydub_loaded:
            tones.beep = patched_beep
            log("Custom beeps active")
        else:
            log("Dependencies unmet; leaving NVDA default beeps")

    def terminate(self):
        tones.beep = original_beep
        log("Custom beep restored to original during terminate")
