# -*- coding: UTF-8 -*-
# Copyright (C) 2009 - 2023 David CM, released under the GPL.
# Author: David CM <dhf360@gmail.com> and others.
# globalPlugins/_ibmttsutils.py

import addonAPIVersion, config, json, os, pickle, wx, zipfile
from os import path
import globalVars, gui, addonHandler
from logHandler import log
from gui.addonGui import promptUserForRestart

addonHandler.initTranslation()

def loadPickle(fileName):
    with open(fileName, "rb") as f:
        return pickle.load(f)

def savePickle(fileName, obj):
    with open(fileName, "wb") as f:
        pickle.dump(obj, f, 4)

def copyFiles(src, dest):
    import installer
    for curSourceDir,subDirs,files in os.walk(src):
        if curSourceDir == src:
            curDestDir=dest
        else:
            curDestDir=path.join(dest,path.relpath(curSourceDir, dest))
        if not path.isdir(curDestDir):
            os.makedirs(curDestDir)
        for f in files:
            sourceFilePath=path.join(curSourceDir,f)
            destFilePath = path.join(curDestDir,f)
            installer.tryCopyFile(sourceFilePath, destFilePath)

def guiCopiFiles(src, dest, title, msg):
    gui.mainFrame.prePopup()
    progressDialog = gui.IndeterminateProgressDialog(gui.mainFrame, title, msg)
    res = True
    while True:
        try:
            gui.ExecAndPump(copyFiles, src, dest)
            break
        except:
            # Translators: a message dialog asking to retry or cancel when copying files.
            message=_("Unable to copy a file. Perhaps it is currently being used by another process or you have run out of disc space on the drive you are copying to.")
            # Translators: the title of a retry cancel dialog when copying files.
            title=_("Error Copying")
            if winUser.MessageBox(None,message,title,winUser.MB_RETRYCANCEL) != winUser.IDRETRY:
                res=False
                log.debugWarning(f"Error when copying files, source {src}, destination {dest}", exc_info=True)
                break
    progressDialog.done()
    del progressDialog
    gui.mainFrame.postPopup()
    return res

def guiInstallAddon(addonPath):
    res = True
    try:
        bundle = addonHandler.AddonBundle(addonPath)
    except:
        log.error(f"Error opening addon update from {addonPath}", exc_info=True)
        gui.messageBox(
            # Translators: The message displayed when an error occurs when opening an add-on package for adding.
            _("Failed to open add-on update file at %s - missing file or invalid file format") % addonPath,
            # Translators: The title of a dialog presented when an error occurs.
            _("Error"),
            wx.OK | wx.ICON_ERROR
        )
        return False
    addonHandler.installAddonBundle(bundle)

def _showAddonRequiresNVDAUpdateDialog(parent, bundle):
    incompatibleMessage = _(
        # Translators: The message displayed when installing an add-on package is prohibited,
        # because it requires a later version of NVDA than is currently installed.
        "Installation of {summary} {version} has been blocked. The minimum NVDA version required for "
        "this add-on is {minimumNVDAVersion}, your current NVDA version is {NVDAVersion}"
    ).format(
        summary=bundle.manifest['summary'],
        version=bundle.manifest['version'],
        minimumNVDAVersion=addonAPIVersion.formatForGUI(bundle.minimumNVDAVersion),
        NVDAVersion=addonAPIVersion.formatForGUI(addonAPIVersion.CURRENT)
    )
    gui.messageBox(
        # Translators: The message displayed when an error occurs when opening an add-on package for adding.
        incompatibleMessage,
        # Translators: The title of a dialog presented when an error occurs.
        _("Add-on not compatible"),
        wx.OK | wx.ICON_ERROR
    )

def _showAddonTooOldDialog(parent, bundle):
    msg = _(
        # Translators: A message informing the user that this addon can not be installed
        # because it is not compatible.
        "Installation of {summary} {version} has been blocked."
        " An updated version of this add-on is required,"
        " the minimum add-on API supported by this version of NVDA is {backCompatToAPIVersion}"
    ).format(
        backCompatToAPIVersion=addonAPIVersion.formatForGUI(addonAPIVersion.BACK_COMPAT_TO),
        **bundle.manifest
    )
    gui.messageBox(
        msg,
        # Translators: The title of the dialog presented when the add-on is too old.
        _("Add-on not compatible"),
        wx.OK | wx.ICON_ERROR
    )

class DonationDialog(gui.nvdaControls.MessageDialog):
    def __init__(self, parent, title, message, donateOptions):
        self.donateOptions = donateOptions
        super().__init__(parent, title, message, dialogType=gui.nvdaControls.MessageDialog.DIALOG_TYPE_WARNING)

    def _addButtons(self, buttonHelper):
        for k in self.donateOptions:
            btn = buttonHelper.addButton(self, label=k['label'], name=k['url'])
            btn.Bind(wx.EVT_BUTTON, self.onDonate)
        cancelBtn = buttonHelper.addButton(self, id=wx.ID_CANCEL, label=_("&Not now"))
        cancelBtn.Bind(wx.EVT_BUTTON, lambda evt: self.EndModal(wx.CANCEL))

    def onDonate(self, evt):
        donateBtn = evt.GetEventObject()
        donateUrl = donateBtn.Name
        os.startfile(donateUrl)
        self.EndModal(wx.OK)

def showDonationsDialog(parentWindow, addonName, donateOptions):
    title = _("Request for contributions to %s") % addonName
    message = _("""Creating add-ons demands substantial time and effort. With limited job prospects in my country, your donations could significantly aid in dedicating more time to developing free plugins for the community.
Your contribution would support the development of this and other free projects.
Would you like to contribute to this cause? Select from our available payment methods below. You will be redirected to the corresponding website to complete your donation.
Thank you for your support and generosity.""")
    return DonationDialog(parentWindow, title,  message, donateOptions).ShowModal()