# pip install wxpython

import wx
import os
import subprocess

class AudioConverterApp(wx.Frame):
    def __init__(self, parent, title):
        super(AudioConverterApp, self).__init__(parent, title=title, size=(500, 400))

        self.init_ui()
        self.Centre()

    def init_ui(self):
        panel = wx.Panel(self)
        vbox = wx.BoxSizer(wx.VERTICAL)

        # Input Directory
        self.input_dir_label = wx.StaticText(panel, label="Input Directory:")
        self.input_dir_picker = wx.DirPickerCtrl(panel, message="Select input directory")
        vbox.Add(self.input_dir_label, flag=wx.LEFT | wx.TOP, border=10)
        vbox.Add(self.input_dir_picker, flag=wx.EXPAND | wx.LEFT | wx.RIGHT | wx.TOP, border=10)

        # Output Directory
        self.output_dir_label = wx.StaticText(panel, label="Output Directory:")
        self.output_dir_picker = wx.DirPickerCtrl(panel, message="Select output directory")
        vbox.Add(self.output_dir_label, flag=wx.LEFT | wx.TOP, border=10)
        vbox.Add(self.output_dir_picker, flag=wx.EXPAND | wx.LEFT | wx.RIGHT | wx.TOP, border=10)

        # Output Format Dropdown
        self.format_label = wx.StaticText(panel, label="Output Format:")
        self.format_choice = wx.Choice(panel, choices=['mp3', 'wav', 'aac', 'flac', 'ogg'])
        self.format_choice.SetSelection(0)  # Default to first choice ('mp3')
        vbox.Add(self.format_label, flag=wx.LEFT | wx.TOP, border=10)
        vbox.Add(self.format_choice, flag=wx.EXPAND | wx.LEFT | wx.RIGHT | wx.TOP, border=10)

        # Convert Button
        convert_button = wx.Button(panel, label='Convert Files')
        convert_button.Bind(wx.EVT_BUTTON, self.on_convert)
        vbox.Add(convert_button, flag=wx.ALIGN_CENTER | wx.TOP, border=20)

        # Result Display
        self.result_label = wx.StaticText(panel, label="", style=wx.ALIGN_CENTER)
        vbox.Add(self.result_label, flag=wx.ALIGN_CENTER | wx.ALL, border=10)

        panel.SetSizer(vbox)

    def on_convert(self, event):
        input_directory = self.input_dir_picker.GetPath()
        output_directory = self.output_dir_picker.GetPath()
        output_format = self.format_choice.GetStringSelection()

        if not input_directory or not output_directory:
            self.result_label.SetLabel("Please select both input and output directories.")
            return

        # Perform conversion
        try:
            self.convert_audio_files(input_directory, output_directory, output_format)
            self.result_label.SetLabel("Files have been successfully converted.")
        except Exception as e:
            self.result_label.SetLabel(f"An error occurred: {e}")

    def convert_audio_files(self, input_directory, output_directory, output_format):
        # Ensure the file format starts with a period and is in lowercase
        output_format = "." + output_format.lower().lstrip('.')

        # Create the output directory if it doesn't exist
        if not os.path.exists(output_directory):
            os.makedirs(output_directory)

        # Iterate over all files in the input directory
        for filename in os.listdir(input_directory):
            # Construct full file paths, assuming any file may be an audio file
            input_file_path = os.path.join(input_directory, filename)
            output_filename = os.path.splitext(filename)[0] + output_format
            output_file_path = os.path.join(output_directory, output_filename)

            # Run the FFMpeg command to convert the file to the specified format
            command = [
                'ffmpeg', '-i', input_file_path,  # Input file
                '-q:a', '2',                      # Set the audio quality (0 = highest, 9 = lowest for MP3)
                output_file_path                 # Output file
            ]

            # Execute the command
            try:
                print(f"Converting {input_file_path} to {output_file_path}")
                subprocess.run(command, check=True)
            except subprocess.CalledProcessError as e:
                print(f"An error occurred while converting {input_file_path}: {e}")

def main():
    app = wx.App()
    frame = AudioConverterApp(None, "Audio File Converter")
    frame.Show()
    app.MainLoop()

if __name__ == "__main__":
    main()