# pip install wxpython openai pillow

import wx
import wx.lib.scrolledpanel as scrolled
import openai
import requests
import threading
from io import BytesIO
from PIL import Image
import os
import sys

def read_api_key():
    """Read the API key from a file called apikey.txt."""
    if os.path.exists('apikey.txt'):
        with open('apikey.txt', 'r') as file:
            api_key = file.read().strip()
            if api_key:
                return api_key
    return None

def write_api_key(api_key):
    """Write the API key to a file called apikey.txt."""
    with open('apikey.txt', 'w') as file:
        file.write(api_key)

def get_api_key():
    """Get API key either from the file or by asking the user."""
    api_key = read_api_key()
    if api_key:
        return api_key

    # If no API key, ask the user through a dialog
    app = wx.App(False)
    dialog = wx.TextEntryDialog(None, "Enter your OpenAI API Key:", "API Key Required", "")
    if dialog.ShowModal() == wx.ID_OK:
        api_key = dialog.GetValue().strip()
        if api_key:
            write_api_key(api_key)
    else:
        # User clicked Cancel, exit the program
        wx.MessageBox("API key entry canceled. Exiting application.", "Information", wx.ICON_INFORMATION)
        sys.exit()

    dialog.Destroy()
    return api_key

class PreviewFrame(wx.Frame):
    def __init__(self, parent, bitmap):
        super().__init__(parent, title="Image Preview", style=wx.DEFAULT_FRAME_STYLE)
        self.bitmap = bitmap
        self.InitUI()

    def InitUI(self):
        panel = wx.Panel(self)
        vbox = wx.BoxSizer(wx.VERTICAL)

        # Create a scrolled panel to display the image (in case it is larger than the screen)
        scrolled_panel = scrolled.ScrolledPanel(panel, style=wx.SUNKEN_BORDER)
        sizer = wx.BoxSizer(wx.VERTICAL)
        self.image_ctrl = wx.StaticBitmap(scrolled_panel, bitmap=self.bitmap)
        sizer.Add(self.image_ctrl, flag=wx.ALL | wx.ALIGN_CENTER, border=10)
        scrolled_panel.SetSizer(sizer)
        scrolled_panel.SetupScrolling()
        vbox.Add(scrolled_panel, proportion=1, flag=wx.EXPAND | wx.ALL, border=10)

        # Close button at the bottom
        close_btn = wx.Button(panel, label="&Close")
        close_btn.Bind(wx.EVT_BUTTON, lambda event: self.Close())
        vbox.Add(close_btn, flag=wx.ALL | wx.ALIGN_CENTER, border=10)

        panel.SetSizer(vbox)

        # Make the frame full-screen
        self.ShowFullScreen(True)

        # Bind key events (ESC to exit)
        self.Bind(wx.EVT_CHAR_HOOK, self.on_key)

    def on_key(self, event):
        if event.GetKeyCode() == wx.WXK_ESCAPE:
            self.Close()
        else:
            event.Skip()

class ImageGeneratorApp(wx.Frame):
    def __init__(self):
        super().__init__(None, title="AI Image Generator", size=(500, 750))
        self.panel = wx.Panel(self)

        # Get the API key
        self.api_key = get_api_key()

        # Set the API key for the OpenAI client
        openai.api_key = self.api_key

        self.image_url = None
        self.image_pil = None  # PIL image for clipboard copying
        self.generated_bitmap = None  # wx.Bitmap of the generated image

        # Main layout
        vbox = wx.BoxSizer(wx.VERTICAL)

        # Prompt input area
        self.prompt_label = wx.StaticText(self.panel, label="Enter a &prompt for the image:")
        vbox.Add(self.prompt_label, flag=wx.ALL | wx.EXPAND, border=10)
        self.prompt_text = wx.TextCtrl(self.panel, style=wx.TE_MULTILINE, size=(400, 60))
        vbox.Add(self.prompt_text, flag=wx.ALL | wx.EXPAND, border=10)

        # Generate button
        self.generate_btn = wx.Button(self.panel, label="&Generate Image")
        self.generate_btn.Bind(wx.EVT_BUTTON, self.on_generate)
        vbox.Add(self.generate_btn, flag=wx.ALL | wx.ALIGN_CENTER, border=10)

        # Loading message (hidden initially)
        self.loading_label = wx.StaticText(self.panel, label="Generating image, please wait...")
        self.loading_label.Hide()
        vbox.Add(self.loading_label, flag=wx.ALL | wx.ALIGN_CENTER, border=10)

        # Buttons for additional actions
        btn_box = wx.BoxSizer(wx.HORIZONTAL)
        self.preview_btn = wx.Button(self.panel, label="&Preview Image")
        self.preview_btn.Bind(wx.EVT_BUTTON, self.on_preview_image)
        self.preview_btn.Hide()
        btn_box.Add(self.preview_btn, flag=wx.ALL | wx.ALIGN_CENTER, border=10)
        self.copy_btn = wx.Button(self.panel, label="&Copy to Clipboard")
        self.copy_btn.Bind(wx.EVT_BUTTON, self.on_copy_to_clipboard)
        self.copy_btn.Hide()
        btn_box.Add(self.copy_btn, flag=wx.ALL | wx.ALIGN_CENTER, border=10)
        self.download_btn = wx.Button(self.panel, label="&Download Image")
        self.download_btn.Bind(wx.EVT_BUTTON, self.on_download)
        self.download_btn.Hide()
        btn_box.Add(self.download_btn, flag=wx.ALL | wx.ALIGN_CENTER, border=10)
        vbox.Add(btn_box, flag=wx.ALL | wx.ALIGN_CENTER, border=10)

        # Reset button to clear the UI state
        self.reset_btn = wx.Button(self.panel, label="&Reset")
        self.reset_btn.Bind(wx.EVT_BUTTON, self.on_reset)
        vbox.Add(self.reset_btn, flag=wx.ALL | wx.ALIGN_CENTER, border=10)
        self.reset_btn.Hide()  # Hide initially

        self.panel.SetSizer(vbox)
        self.Show()

    def on_generate(self, event):
        """Initiates the image generation process."""
        prompt = self.prompt_text.GetValue().strip()
        if not prompt:
            wx.MessageBox("Please enter a prompt before generating an image.", "Error", wx.ICON_ERROR)
            return

        # Hide all other UI components
        self.prompt_label.Hide()
        self.prompt_text.Hide()
        self.generate_btn.Hide()  # Also hide the button completely
        self.preview_btn.Hide()
        self.copy_btn.Hide()
        self.download_btn.Hide()
        self.reset_btn.Hide()

        # Show only the loading message
        self.loading_label.SetLabel("Generating image, please wait...")
        self.loading_label.Show()
        self.panel.Layout()

        # Set focus to loading label
        self.loading_label.SetFocus()

        # Start image generation in a separate thread to keep the UI responsive
        threading.Thread(target=self.generate_image, args=(prompt,), daemon=True).start()

    def generate_image(self, prompt):
        """Calls the OpenAI API to generate an image from the prompt."""
        try:
            response = openai.images.generate(
                model="dall-e-3",
                prompt=prompt,
                n=1,
                size="1024x1024"
            )
            self.image_url = response.data[0].url
            wx.CallAfter(self.on_generation_success)
        except Exception as e:
            wx.CallAfter(self.on_generation_failure, str(e))

    def on_generation_success(self):
        """Updates the UI after a successful image generation."""
        self.loading_label.SetLabel("Image generated!")
        # Download the image data
        response = requests.get(self.image_url)
        img_data = BytesIO(response.content)
        self.image_pil = Image.open(img_data)  # Store the PIL image for clipboard copying

        # Convert the downloaded image data to a wx.Bitmap
        img_data.seek(0)  # Reset pointer to the beginning
        wx_image = wx.Image(img_data)
        self.generated_bitmap = wx_image.ConvertToBitmap()

        # Reveal the additional action and reset buttons
        self.preview_btn.Show()
        self.copy_btn.Show()
        self.download_btn.Show()
        self.reset_btn.Show()
        self.panel.Layout()

        # Set focus to the first visible button (preview)
        self.preview_btn.SetFocus()

    def on_generation_failure(self, error_message):
        """Handles errors during image generation."""
        wx.MessageBox(f"An error occurred: {error_message}", "Error", wx.ICON_ERROR)

        # Restore the prompt box and hide the loading message
        self.prompt_label.Show()
        self.prompt_text.Show()

        # Re-enable the generate button
        self.generate_btn.Show()

        self.loading_label.Hide()
        self.panel.Layout()

        # Set focus back to the generate button
        self.generate_btn.SetFocus()

    def on_preview_image(self, event):
        """Opens a full-screen preview dialog displaying the generated image."""
        if not self.generated_bitmap:
            wx.MessageBox("No image available to preview.", "Error", wx.ICON_ERROR)
            return
        preview = PreviewFrame(None, self.generated_bitmap)
        preview.Show()

    def on_copy_to_clipboard(self, event):
        """Copies the generated image to the clipboard."""
        if not self.image_pil:
            wx.MessageBox("No image available to copy.", "Error", wx.ICON_ERROR)
            return
        width, height = self.image_pil.size
        # Convert the PIL image to a wx.Bitmap (using RGBA for proper transparency)
        image_wx = wx.Bitmap.FromBufferRGBA(width, height, self.image_pil.convert("RGBA").tobytes())
        if wx.TheClipboard.Open():
            wx.TheClipboard.Clear()
            wx.TheClipboard.SetData(wx.BitmapDataObject(image_wx))
            wx.TheClipboard.Close()
            wx.MessageBox("Image copied to clipboard.", "Success", wx.ICON_INFORMATION)
        else:
            wx.MessageBox("Failed to open clipboard.", "Error", wx.ICON_ERROR)

    def on_download(self, event):
        """Opens a file save dialog to download the generated image."""
        if not self.image_url:
            wx.MessageBox("No image available to download.", "Error", wx.ICON_ERROR)
            return
        with wx.FileDialog(self, "Save Image", wildcard="PNG files (*.png)|*.png",
                           style=wx.FD_SAVE | wx.FD_OVERWRITE_PROMPT) as fileDialog:
            if fileDialog.ShowModal() == wx.ID_CANCEL:
                return  # User cancelled the save operation
            file_path = fileDialog.GetPath()
            try:
                response = requests.get(self.image_url)
                with open(file_path, "wb") as file:
                    file.write(response.content)
                wx.MessageBox(f"Image saved as {file_path}.", "Success", wx.ICON_INFORMATION)
            except Exception as e:
                wx.MessageBox(f"Failed to save image: {str(e)}", "Error", wx.ICON_ERROR)

    def on_reset(self, event):
        """Reset the application to its initial state."""
        # Clear all fields and reset the state
        self.prompt_text.SetValue("")
        self.image_url = None
        self.image_pil = None
        self.generated_bitmap = None

        # Hide action buttons and reset text
        self.preview_btn.Hide()
        self.copy_btn.Hide()
        self.download_btn.Hide()
        self.reset_btn.Hide()
        self.prompt_label.Show()
        self.prompt_text.Show()
        self.generate_btn.Show()
        self.generate_btn.Enable()
        self.loading_label.SetLabel("Generating image, please wait...")
        self.loading_label.Hide()
        self.panel.Layout()

        # Set focus back to the prompt text
        self.prompt_text.SetFocus()

if __name__ == "__main__":
    app = wx.App(False)
    ImageGeneratorApp()
    app.MainLoop()